import { Head } from '@inertiajs/react';
import { useState } from 'react';
import { type Plan } from '@/hooks/usePlans';

interface SubscribeProps {
    plan: Plan;
    checkoutUrl: string;
}

export default function Subscribe({ plan, checkoutUrl }: SubscribeProps) {
    const [isLoading, setIsLoading] = useState(false);

    const handleCheckout = () => {
        setIsLoading(true);
        window.location.href = checkoutUrl;
    };

    return (
        <>
            <Head title={`Subscribe to ${plan.name}`} />
            
            <div className="max-w-3xl mx-auto px-4 py-12">
                <div className="mb-8 text-center">
                    <h1 className="text-3xl font-bold mb-2">Subscribe to {plan.name}</h1>
                    <p className="text-gray-600 dark:text-gray-400">
                        You're just one step away from accessing all the great features
                    </p>
                </div>
                
                <div className="bg-white dark:bg-gray-900 rounded-lg shadow-md overflow-hidden border border-gray-200 dark:border-gray-800">
                    <div className="p-6">
                        <div className="flex justify-between items-start mb-6">
                            <div>
                                <h2 className="text-xl font-semibold">{plan.name}</h2>
                                <p className="text-gray-600 dark:text-gray-400 mt-1">{plan.description}</p>
                            </div>
                            <div className="text-right">
                                <div className="text-2xl font-bold">${(plan.price_cents / 100).toFixed(2)}</div>
                                <div className="text-sm text-gray-500 dark:text-gray-400">
                                    /{plan.billing_interval === 'monthly' ? 'month' : 'year'}
                                </div>
                            </div>
                        </div>
                        
                        <div className="border-t border-gray-200 dark:border-gray-800 pt-6 mb-6">
                            <h3 className="font-medium mb-3">What's included:</h3>
                            <ul className="space-y-3">
                                {Array.isArray(plan.features) && plan.features.map((feature: string, index: number) => (
                                    <li key={index} className="flex items-start">
                                        <svg
                                            xmlns="http://www.w3.org/2000/svg"
                                            className="h-5 w-5 text-green-500 shrink-0 mr-2"
                                            viewBox="0 0 20 20"
                                            fill="currentColor"
                                        >
                                            <path
                                                fillRule="evenodd"
                                                d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"
                                                clipRule="evenodd"
                                            />
                                        </svg>
                                        <span className="text-sm text-gray-700 dark:text-gray-300">{feature}</span>
                                    </li>
                                ))}
                            </ul>
                        </div>
                        
                        <div className="flex flex-col space-y-4">
                            <button
                                onClick={handleCheckout}
                                disabled={isLoading}
                                className="w-full py-3 px-4 bg-indigo-600 hover:bg-indigo-700 text-white rounded-md font-medium focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 dark:focus:ring-offset-gray-900 transition-colors disabled:opacity-70"
                            >
                                {isLoading ? (
                                    <span className="flex items-center justify-center">
                                        <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                            <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                            <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                        </svg>
                                        Redirecting to checkout...
                                    </span>
                                ) : (
                                    'Proceed to Checkout'
                                )}
                            </button>
                            <p className="text-center text-sm text-gray-500 dark:text-gray-400">
                                You'll be redirected to LemonSqueezy's secure checkout page
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </>
    );
}
