<?php

namespace App\Http\Middleware;

use App\Models\Plan;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class FeatureRestrictionMiddleware
{
    /**
     * Handle an incoming request.
     * Check if the user has access to a specific feature based on their plan.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     * @param  string  $feature  The feature to check access for (e.g., 'feature1', 'feature5')
     * @param  string  $requiredPlanType  The plan type required ('starter' or 'pro')
     */
    public function handle(Request $request, Closure $next, string $feature, string $requiredPlanType): Response
    {
        $user = Auth::user();

        if (!$user) {
            return redirect()->route('login');
        }

        // Check if the user has an active subscription
        $activeSubscription = $user->subscriptions()->get()->first(function ($subscription) {
            return $subscription->active();
        });

        if (!$activeSubscription) {
            return redirect()->route('subscription.index')
                ->with('error', 'You need an active subscription to access this feature.');
        }

        $plan = Plan::where('lemonsqueezy_price_id', $activeSubscription->variant_id)
            ->where('lemonsqueezy_product_id', $activeSubscription->product_id)
            ->first();

        if (!$plan) {
            return redirect()->route('subscription.index')
                ->with('error', 'Your subscription plan is not recognized. Please contact support.');
        }

        // Get the plan slug from the active subscription
        $planSlug = $plan->slug;

        // Check if the plan type matches the required type
        $hasRequiredPlanType = false;

        if ($requiredPlanType === 'starter') {
            // For starter features, both starter and pro plans have access
            $hasRequiredPlanType = str_contains($planSlug, 'starter') || str_contains($planSlug, 'pro');
        } elseif ($requiredPlanType === 'pro') {
            // For pro features, only pro plans have access
            $hasRequiredPlanType = str_contains($planSlug, 'pro');
        }

        if (!$hasRequiredPlanType) {
            $planName = ucfirst($requiredPlanType);
            return redirect()->route('subscription.index')
                ->with('error', "This feature requires a {$planName} plan or higher. Please upgrade your subscription.");
        }

        return $next($request);
    }
}
