<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class WebhookController extends Controller
{
    /**
     * Handle LemonSqueezy webhook requests.
     */
    public function handleLemonSqueezyWebhook(Request $request): Response
    {
        // Verify webhook signature
        if (!$this->isValidSignature($request)) {
            Log::error('Invalid webhook signature', [
                'headers' => $request->headers->all(),
                'payload' => $request->all(),
            ]);
            return response('Invalid signature', 401);
        }

        $payload = $request->all();
        $eventName = $request->header('X-Event-Name');

        // Log the webhook event
        Log::info('LemonSqueezy webhook received', [
            'event' => $eventName,
            'payload' => $payload,
        ]);

        // Process the webhook based on event type
        try {
            match ($eventName) {
                'subscription_created' => $this->handleSubscriptionCreated($payload),
                'subscription_updated' => $this->handleSubscriptionUpdated($payload),
                'subscription_cancelled' => $this->handleSubscriptionCancelled($payload),
                'subscription_resumed' => $this->handleSubscriptionResumed($payload),
                'subscription_expired' => $this->handleSubscriptionExpired($payload),
                'invoice_paid' => $this->handleInvoicePaid($payload),
                default => $this->handleUnknownEvent($eventName, $payload),
            };

            return response('Webhook processed', 200);
        } catch (\Exception $e) {
            Log::error('Error processing webhook', [
                'event' => $eventName,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response('Error processing webhook', 500);
        }
    }

    /**
     * Verify that the webhook request is coming from LemonSqueezy.
     */
    protected function isValidSignature(Request $request): bool
    {
        $signature = $request->header('X-Signature');
        $secret = config('lemonsqueezy.webhook_secret');

        if (!$secret || !$signature) {
            return false;
        }

        $computedSignature = hash_hmac('sha256', $request->getContent(), $secret);

        return hash_equals($computedSignature, $signature);
    }

    /**
     * Handle subscription_created event.
     */
    protected function handleSubscriptionCreated(array $payload): void
    {
        $data = $payload['data']['attributes'];
        $meta = $payload['meta'];

        $user = User::where('email', $meta['custom_data']['email'] ?? null)->first();

        if (!$user) {
            Log::error('User not found for subscription', [
                'subscription_id' => $data['id'],
                'email' => $meta['custom_data']['email'] ?? null,
            ]);
            return;
        }

        $planSlug = Str::slug($data['variant_name']);
        $endsAt = null;

        if ($data['ends_at']) {
            $endsAt = Carbon::parse($data['ends_at']);
        }

        $trialEndsAt = null;
        if ($data['trial_ends_at']) {
            $trialEndsAt = Carbon::parse($data['trial_ends_at']);
        }

        $user->update([
            'is_subscribed' => true,
            'subscription_id' => $data['id'],
            'plan_name' => $planSlug,
            'ends_at' => $endsAt,
            'trial_ends_at' => $trialEndsAt,
            'lemon_customer_id' => $data['customer_id'],
        ]);

        Log::info('User subscribed', [
            'user_id' => $user->id,
            'subscription_id' => $data['id'],
            'plan' => $planSlug,
        ]);
    }

    /**
     * Handle subscription_updated event.
     */
    protected function handleSubscriptionUpdated(array $payload): void
    {
        $data = $payload['data']['attributes'];
        $subscriptionId = $data['id'];

        $user = User::where('subscription_id', $subscriptionId)->first();

        if (!$user) {
            Log::error('User not found for subscription update', [
                'subscription_id' => $subscriptionId,
            ]);
            return;
        }

        $endsAt = null;
        if ($data['ends_at']) {
            $endsAt = Carbon::parse($data['ends_at']);
        }

        $trialEndsAt = null;
        if ($data['trial_ends_at']) {
            $trialEndsAt = Carbon::parse($data['trial_ends_at']);
        }

        $user->update([
            'ends_at' => $endsAt,
            'trial_ends_at' => $trialEndsAt,
        ]);

        Log::info('Subscription updated', [
            'user_id' => $user->id,
            'subscription_id' => $subscriptionId,
        ]);
    }

    /**
     * Handle subscription_cancelled event.
     */
    protected function handleSubscriptionCancelled(array $payload): void
    {
        $data = $payload['data']['attributes'];
        $subscriptionId = $data['id'];

        $user = User::where('subscription_id', $subscriptionId)->first();

        if (!$user) {
            Log::error('User not found for subscription cancellation', [
                'subscription_id' => $subscriptionId,
            ]);
            return;
        }

        $endsAt = null;
        if ($data['ends_at']) {
            $endsAt = Carbon::parse($data['ends_at']);
        }

        $user->cancelSubscription($endsAt);

        Log::info('Subscription cancelled', [
            'user_id' => $user->id,
            'subscription_id' => $subscriptionId,
            'ends_at' => $endsAt,
        ]);
    }

    /**
     * Handle subscription_resumed event.
     */
    protected function handleSubscriptionResumed(array $payload): void
    {
        $data = $payload['data']['attributes'];
        $subscriptionId = $data['id'];

        $user = User::where('subscription_id', $subscriptionId)->first();

        if (!$user) {
            Log::error('User not found for subscription resumption', [
                'subscription_id' => $subscriptionId,
            ]);
            return;
        }

        $user->update([
            'is_subscribed' => true,
            'ends_at' => null,
        ]);

        Log::info('Subscription resumed', [
            'user_id' => $user->id,
            'subscription_id' => $subscriptionId,
        ]);
    }

    /**
     * Handle subscription_expired event.
     */
    protected function handleSubscriptionExpired(array $payload): void
    {
        $data = $payload['data']['attributes'];
        $subscriptionId = $data['id'];

        $user = User::where('subscription_id', $subscriptionId)->first();

        if (!$user) {
            Log::error('User not found for subscription expiration', [
                'subscription_id' => $subscriptionId,
            ]);
            return;
        }

        $user->markSubscriptionAsExpired();

        Log::info('Subscription expired', [
            'user_id' => $user->id,
            'subscription_id' => $subscriptionId,
        ]);
    }

    /**
     * Handle invoice_paid event.
     */
    protected function handleInvoicePaid(array $payload): void
    {
        $data = $payload['data']['attributes'];
        $orderId = $data['order_id'];

        // Log the invoice payment
        Log::info('Invoice paid', [
            'order_id' => $orderId,
            'total' => $data['total'],
            'status' => $data['status'],
        ]);

        // Additional logic for invoice payment can be added here
    }

    /**
     * Handle unknown events.
     */
    protected function handleUnknownEvent(string $eventName, array $payload): void
    {
        Log::info('Unhandled webhook event', [
            'event' => $eventName,
            'payload' => $payload,
        ]);
    }
}
