<?php

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Illuminate\Validation\ValidationException;
use Livewire\Attributes\Title;
use Livewire\Volt\Component;

new #[Title('Password')] class extends Component {
    public string $current_password = '';
    public string $password = '';
    public string $password_confirmation = '';
    public bool $updated = false;

    /**
     * Update the password for the currently authenticated user.
     */
    public function updatePassword(): void
    {
        $user = Auth::user();

        try {
            $rules = [
                'password' => ['required', 'string', Password::min(15)->uncompromised()->letters()->numbers()->symbols(), 'confirmed'],
            ];

            if (! $user?->force_password_reset) {
                $rules['current_password'] = ['required', 'string', 'current_password'];
            }

            $validated = $this->validate($rules);
        } catch (ValidationException $e) {
            $this->reset('current_password', 'password', 'password_confirmation');

            throw $e;
        }

        $user?->update([
            'password' => Hash::make($validated['password']),
            'force_password_reset' => false,
        ]);

        $this->reset('current_password', 'password', 'password_confirmation');
        
        $this->updated = true;
        
        $this->dispatch('password-updated');
    }
}; ?>

<div>
    @include('partials.settings-heading')

    <section class="w-full">
        <x-settings.layout :heading="__('Update password')" :subheading="__('Ensure your account is using a long, random password to stay secure')">
            @if (Auth::user()?->force_password_reset)
                <div class="rounded-md bg-amber-50 p-4 text-sm text-amber-800">
                    {{ __('An employee of :app has set you a temporary password.', ['app' => config('app.name')]) }}<br>
                    {{ __('Please reset your password before you can continue.') }}
                </div>
            @endif
            <form wire:submit="updatePassword" class="mt-6 space-y-6">
                @unless (Auth::user()?->force_password_reset)
                    <flux:input
                        wire:model="current_password"
                        :label="__('Current password')"
                        type="password"
                        required
                        autocomplete="current-password"
                    />
                @endunless
                <flux:input
                    wire:model="password"
                    :label="__('New password')"
                    type="password"
                    required
                    autocomplete="new-password"
                />
                <flux:input
                    wire:model="password_confirmation"
                    :label="__('Confirm Password')"
                    type="password"
                    required
                    autocomplete="new-password"
                />

                <div class="flex items-center gap-4">
                    <flux:button class="!bg-indigo-700 hover:!bg-indigo-500 !text-white" type="submit">
                        {{ __('Save') }}
                    </flux:button>

                    @if ($updated)
                        <span 
                            x-data="{ show: true }"
                            x-show="show"
                            x-init="setTimeout(() => show = false, 2000)"
                            x-transition
                            class="text-sm text-green-600 font-medium"
                        >
                            {{ __('Updated.') }}
                        </span>
                    @endif
                </div>
            </form>
        </x-settings.layout>
    </section>
</div>