<div>
    @push('scripts')
        <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
        <script>
            (function () {
                const chartInstances = {};

                function getChartPayload() {
                    const payloadEl = document.getElementById('sla-dashboard-chart-data');
                    if (!payloadEl) {
                        return null;
                    }

                    try {
                        return JSON.parse(payloadEl.dataset.chart || '{}');
                    } catch (error) {
                        console.warn('SLA Dashboard: failed to parse chart payload', error);
                        return null;
                    }
                }

                function destroyChart(key) {
                    if (chartInstances[key]) {
                        chartInstances[key].destroy();
                        delete chartInstances[key];
                    }
                }

                function renderCharts() {
                    if (typeof Chart === 'undefined') {
                        return;
                    }

                    const chartData = getChartPayload();
                    if (!chartData) {
                        toggleEmptyState('departmentChartEmpty', true);
                        toggleEmptyState('priorityChartEmpty', true);
                        return;
                    }

                    const deptCtx = document.getElementById('departmentChart');
                    const hasDepartmentData = Array.isArray(chartData.departmentLabels) && chartData.departmentLabels.length > 0;

                    if (deptCtx) {
                        if (!hasDepartmentData) {
                            destroyChart('department');
                            toggleEmptyState('departmentChartEmpty', true);
                        } else {
                            toggleEmptyState('departmentChartEmpty', false);
                            destroyChart('department');
                            chartInstances.department = new Chart(deptCtx, {
                                type: 'bar',
                                data: {
                                    labels: chartData.departmentLabels,
                                    datasets: [{
                                        label: 'Compliance Rate (%)',
                                        data: chartData.departmentCompliance ?? [],
                                        backgroundColor: 'rgba(59, 130, 246, 0.5)',
                                        borderColor: 'rgba(59, 130, 246, 1)',
                                        borderWidth: 1
                                    }]
                                },
                                options: {
                                    responsive: true,
                                    maintainAspectRatio: false,
                                    scales: {
                                        y: {
                                            beginAtZero: true,
                                            max: 100,
                                            ticks: {
                                                callback: value => value + '%'
                                            }
                                        }
                                    }
                                }
                            });
                        }
                    }

                    const priCtx = document.getElementById('priorityChart');
                    const priorityValues = chartData.priorityBreached ?? [];
                    const hasPriorityData = Array.isArray(chartData.priorityLabels)
                        && chartData.priorityLabels.length > 0
                        && priorityValues.some(value => Number(value) > 0);

                    if (priCtx) {
                        if (!hasPriorityData) {
                            destroyChart('priority');
                            toggleEmptyState('priorityChartEmpty', true);
                        } else {
                            toggleEmptyState('priorityChartEmpty', false);
                            destroyChart('priority');
                            chartInstances.priority = new Chart(priCtx, {
                                type: 'doughnut',
                                data: {
                                    labels: chartData.priorityLabels,
                                    datasets: [{
                                        label: 'Breached Tickets',
                                        data: priorityValues,
                                        backgroundColor: [
                                            'rgba(239, 68, 68, 0.5)',
                                            'rgba(251, 191, 36, 0.5)',
                                            'rgba(34, 197, 94, 0.5)',
                                            'rgba(59, 130, 246, 0.5)'
                                        ],
                                        borderColor: [
                                            'rgba(239, 68, 68, 1)',
                                            'rgba(251, 191, 36, 1)',
                                            'rgba(34, 197, 94, 1)',
                                            'rgba(59, 130, 246, 1)'
                                        ],
                                        borderWidth: 1
                                    }]
                                },
                                options: {
                                    responsive: true,
                                    maintainAspectRatio: false,
                                    plugins: {
                                        legend: {
                                            position: 'bottom'
                                        }
                                    }
                                }
                            });
                        }
                    }
                }

                document.addEventListener('livewire:initialized', () => {
                    renderCharts();

                    Livewire.hook('message.processed', () => {
                        renderCharts();
                    });
                });

                function toggleEmptyState(id, show) {
                    const el = document.getElementById(id);
                    if (!el) {
                        return;
                    }

                    el.classList.toggle('hidden', !show);
                }
            })();
        </script>
    @endpush

    <div class="container mx-auto px-8 pb-8">
        <div id="sla-dashboard-chart-data" data-chart='@json($chartData)' class="hidden"></div>
        <!-- Header -->
            <div class="flex items-center justify-between mb-6">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900 dark:text-white">SLA Tracking Dashboard</h1>
                    <p class="text-gray-600 dark:text-gray-400 mt-1">Monitor and manage Service Level Agreement compliance</p>
                </div>
                <button wire:click="loadDashboardData" wire:loading.attr="disabled"
                    class="px-4 py-2 bg-indigo-600 hover:bg-indigo-700 text-white rounded-lg transition disabled:opacity-50">
                    <svg wire:loading class="inline w-4 h-4 mr-2 animate-spin" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
                    </svg>
                    <span wire:loading.remove>Refresh</span>
                    <span wire:loading>Refreshing...</span>
                </button>
            </div>

            <!-- Statistics Cards -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
                <!-- Total Tickets -->
                <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600 dark:text-gray-400">Total Tickets</p>
                            <p class="text-3xl font-bold text-gray-900 dark:text-white mt-2">
                                {{ $stats['total_tickets'] ?? 0 }}
                            </p>
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Last 30 days</p>
                        </div>
                        <div class="bg-blue-100 dark:bg-blue-900 rounded-full p-3">
                            <svg class="w-8 h-8 text-blue-600 dark:text-blue-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
                            </svg>
                        </div>
                    </div>
                </div>

                <!-- SLA Met -->
                <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600 dark:text-gray-400">SLA Met</p>
                            <p class="text-3xl font-bold text-green-600 dark:text-green-400 mt-2">
                                {{ $stats['sla_met'] ?? 0 }}
                            </p>
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Compliance achieved</p>
                        </div>
                        <div class="bg-green-100 dark:bg-green-900 rounded-full p-3">
                            <svg class="w-8 h-8 text-green-600 dark:text-green-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                        </div>
                    </div>
                </div>

                <!-- At Risk -->
                <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600 dark:text-gray-400">At Risk</p>
                            <p class="text-3xl font-bold text-yellow-600 dark:text-yellow-400 mt-2">
                                {{ $stats['at_risk'] ?? 0 }}
                            </p>
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Needs attention</p>
                        </div>
                        <div class="bg-yellow-100 dark:bg-yellow-900 rounded-full p-3">
                            <svg class="w-8 h-8 text-yellow-600 dark:text-yellow-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                            </svg>
                        </div>
                    </div>
                </div>

                <!-- Breached -->
                <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600 dark:text-gray-400">Breached</p>
                            <p class="text-3xl font-bold text-red-600 dark:text-red-400 mt-2">
                                {{ $stats['sla_breached'] ?? 0 }}
                            </p>
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">SLA violations</p>
                        </div>
                        <div class="bg-red-100 dark:bg-red-900 rounded-full p-3">
                            <svg class="w-8 h-8 text-red-600 dark:text-red-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Compliance Rate -->
            <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6 mb-8">
                <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">Overall SLA Compliance Rate</h3>
                <div class="flex items-center">
                    <div class="flex-1">
                        <div class="relative pt-1">
                            <div class="overflow-hidden h-6 text-xs flex rounded-full bg-gray-200 dark:bg-gray-700">
                                <div style="width:{{ $stats['compliance_rate'] ?? 0 }}%"
                                    class="shadow-none flex flex-col text-center whitespace-nowrap text-white justify-center transition-all duration-500
                                    {{ ($stats['compliance_rate'] ?? 0) >= 90 ? 'bg-green-500' : (($stats['compliance_rate'] ?? 0) >= 70 ? 'bg-yellow-500' : 'bg-red-500') }}">
                                </div>
                            </div>
                        </div>
                    </div>
                    <span class="ml-4 text-3xl font-bold
                        {{ ($stats['compliance_rate'] ?? 0) >= 90 ? 'text-green-600 dark:text-green-400' : (($stats['compliance_rate'] ?? 0) >= 70 ? 'text-yellow-600 dark:text-yellow-400' : 'text-red-600 dark:text-red-400') }}">
                        {{ number_format($stats['compliance_rate'] ?? 0, 1) }}%
                    </span>
                </div>
            </div>

            <!-- Charts -->
            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
                <!-- Department Performance Chart -->
                <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                    <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">Department Compliance</h3>
                    <div class="relative h-64">
                        <canvas id="departmentChart" height="250"></canvas>
                        <div id="departmentChartEmpty" class="absolute inset-0 flex items-center justify-center text-sm text-gray-500 dark:text-gray-400 hidden">
                            Not enough data yet
                        </div>
                    </div>
                </div>

                <!-- Priority Breach Chart -->
                <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                    <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">Breaches by Priority</h3>
                    <div class="relative h-64">
                        <canvas id="priorityChart" height="250"></canvas>
                        <div id="priorityChartEmpty" class="absolute inset-0 flex items-center justify-center text-sm text-gray-500 dark:text-gray-400 hidden">
                            Not enough data yet
                        </div>
                    </div>
                </div>
            </div>

            <!-- Ticket Lists -->
            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                <!-- At Risk Tickets -->
                <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                    <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">At Risk Tickets</h3>
                    @if($atRiskTickets->count() > 0)
                    <div class="space-y-3">
                        @foreach($atRiskTickets as $ticket)
                        <a href="/tickets/{{ $ticket->ticket_id }}" wire:navigate
                            class="block p-3 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg hover:bg-yellow-100 dark:hover:bg-yellow-900/30 transition">
                            <div class="flex items-start justify-between">
                                <div class="flex-1">
                                    <p class="font-medium text-gray-900 dark:text-white text-sm">
                                        #{{ $ticket->ticket_id }} - {{ Str::limit($ticket->subject, 40) }}
                                    </p>
                                    <p class="text-xs text-gray-600 dark:text-gray-400 mt-1">
                                        {{ $ticket->department->department_name ?? 'N/A' }} • {{ ucfirst($ticket->priority->label()) }}
                                    </p>
                                </div>
                                <span class="ml-2 text-xs text-yellow-700 dark:text-yellow-300">
                                    {{ \Carbon\Carbon::parse($ticket->sla_first_response_due_at ?? $ticket->sla_resolution_due_at)->diffForHumans() }}
                                </span>
                            </div>
                        </a>
                        @endforeach
                    </div>
                    @else
                    <p class="text-center text-gray-500 dark:text-gray-400 py-8">No tickets at risk</p>
                    @endif
                </div>

                <!-- Breached Tickets -->
                <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                    <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">Breached SLAs</h3>
                    @if($breachedTickets->count() > 0)
                    <div class="space-y-3">
                        @foreach($breachedTickets as $ticket)
                        <a href="/tickets/{{ $ticket->ticket_id }}" wire:navigate
                            class="block p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg hover:bg-red-100 dark:hover:bg-red-900/30 transition">
                            <div class="flex items-start justify-between">
                                <div class="flex-1">
                                    <p class="font-medium text-gray-900 dark:text-white text-sm">
                                        #{{ $ticket->ticket_id }} - {{ Str::limit($ticket->subject, 40) }}
                                    </p>
                                    <p class="text-xs text-gray-600 dark:text-gray-400 mt-1">
                                        {{ $ticket->department->department_name ?? 'N/A' }} • {{ ucfirst($ticket->priority->label()) }}
                                    </p>
                                </div>
                                <span class="ml-2 px-2 py-1 text-xs font-medium bg-red-100 dark:bg-red-900 text-red-800 dark:text-red-200 rounded">
                                    Breached
                                </span>
                            </div>
                        </a>
                        @endforeach
                    </div>
                    @else
                    <p class="text-center text-gray-500 dark:text-gray-400 py-8">No breached SLAs</p>
                    @endif
                </div>
            </div>
    </div>

</div>
