<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class Settings extends Model
{
    protected $fillable = [
        'key',
        'value',
        'type',
        'description',
    ];

    /**
     * Get a setting value by key with caching
     */
    public static function get(string $key, $default = null)
    {
        try {
            return Cache::remember("setting.{$key}", now()->addHours(24), function () use ($key, $default) {
                $setting = self::where('key', $key)->first();

                if (!$setting) {
                    return $default;
                }

                return self::castValue($setting->value, $setting->type);
            });
        } catch (\Exception $e) {
            // Table doesn't exist yet or database error - return default
            return $default;
        }
    }

    /**
     * Set a setting value by key and clear cache
     */
    public static function set(string $key, $value, string $type = 'string', ?string $description = null)
    {
        try {
            $setting = self::updateOrCreate(
                ['key' => $key],
                [
                    'value' => $value,
                    'type' => $type,
                    'description' => $description,
                ]
            );

            // Clear cache
            Cache::forget("setting.{$key}");

            return $setting;
        } catch (\Exception $e) {
            // Table doesn't exist yet or database error
            throw $e;
        }
    }

    /**
     * Cast value based on type
     */
    protected static function castValue($value, string $type)
    {
        return match ($type) {
            'boolean' => (bool) $value,
            'integer' => (int) $value,
            'float' => (float) $value,
            'json' => json_decode($value, true),
            default => $value,
        };
    }

    /**
     * Clear cache for a specific setting
     */
    public static function clearCache(string $key): void
    {
        try {
            Cache::forget("setting.{$key}");
        } catch (\Exception $e) {
            // Ignore cache errors
        }
    }

    /**
     * Clear all settings cache
     */
    public static function clearAllCache(): void
    {
        try {
            Cache::flush();
        } catch (\Exception $e) {
            // Ignore cache errors
        }
    }
}
