<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Enums\{Type};

class Categories extends Model
{
    protected $fillable = [
        'category_name',
        'uri',
        'type',
        'display',
        'parent_id',
        'sort_order',
    ];

    protected $casts = [
        'type' => Type::class,
    ];

    /**
     * Get all announcements for this category
     */
    public function announcements()
    {
        return $this->hasMany(Announcements::class, 'parent_category');
    }

    /**
     * Get all knowledgebase articles for this category
     */
    public function knowledgebase()
    {
        return $this->hasMany(\App\Extensions\Installed\Knowledgebase\Models\Article::class, 'parent_category');
    }

    /**
     * Get the parent category
     */
    public function parent()
    {
        return $this->belongsTo(Categories::class, 'parent_id');
    }

    /**
     * Get all child categories
     */
    public function children()
    {
        return $this->hasMany(Categories::class, 'parent_id')->orderBy('sort_order');
    }

    /**
     * Check if this category has any children
     */
    public function hasChildren()
    {
        return $this->children()->exists();
    }

    /**
     * Scope to get only top-level categories (no parent)
     */
    public function scopeTopLevel($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * Scope to order categories by sort_order
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('category_name');
    }

    /**
     * Get the depth level of this category (0 = top level, 1 = child, 2 = grandchild, etc.)
     */
    public function getDepth()
    {
        $depth = 0;
        $current = $this;

        while ($current->parent_id) {
            $depth++;
            $current = $current->parent;

            // Prevent infinite loops
            if ($depth > 10) {
                break;
            }
        }

        return $depth;
    }

    /**
     * Get all ancestor categories (parent, grandparent, etc.)
     */
    public function ancestors()
    {
        $ancestors = collect();
        $current = $this->parent;

        while ($current) {
            $ancestors->push($current);
            $current = $current->parent;

            // Prevent infinite loops
            if ($ancestors->count() > 10) {
                break;
            }
        }

        return $ancestors;
    }

    /**
     * Check if this category can have the specified parent
     *
     * @param int|null $parentId The ID of the potential parent
     * @param int $maxDepth Maximum allowed depth (default 3)
     * @return array ['valid' => bool, 'error' => string|null]
     */
    public function canHaveParent($parentId, $maxDepth = 3)
    {
        // No parent is always allowed
        if (!$parentId) {
            return ['valid' => true, 'error' => null];
        }

        // Cannot be its own parent
        if ($this->id && $parentId == $this->id) {
            return ['valid' => false, 'error' => 'A category cannot be its own parent.'];
        }

        $parent = Categories::find($parentId);

        if (!$parent) {
            return ['valid' => false, 'error' => 'Parent category not found.'];
        }

        // Check if types match
        $parentType = $parent->type instanceof \BackedEnum ? $parent->type->value : $parent->type;
        $thisType = $this->type instanceof \BackedEnum ? $this->type->value : $this->type;

        if ($parentType !== $thisType) {
            return ['valid' => false, 'error' => 'Parent category must be the same type.'];
        }

        // Check if the parent's depth would exceed max depth
        if ($parent->getDepth() >= $maxDepth - 1) {
            return ['valid' => false, 'error' => "Maximum nesting depth of {$maxDepth} levels exceeded."];
        }

        // Check if the parent is a descendant of this category (would create a loop)
        if ($this->id) {
            $current = $parent;
            $checked = 0;

            while ($current->parent_id) {
                if ($current->parent_id == $this->id) {
                    return ['valid' => false, 'error' => 'Cannot set a descendant as parent (would create a loop).'];
                }

                $current = $current->parent;
                $checked++;

                // Prevent infinite loops
                if ($checked > 10) {
                    break;
                }
            }
        }

        return ['valid' => true, 'error' => null];
    }

    /**
     * Get the breadcrumb path for this category (e.g., "Parent > Child > This")
     */
    public function getBreadcrumbPath()
    {
        $path = [];
        $current = $this;

        while ($current) {
            array_unshift($path, $current->category_name);
            $current = $current->parent;

            // Prevent infinite loops
            if (count($path) > 10) {
                break;
            }
        }

        return implode(' > ', $path);
    }
}
