<?php

namespace App\Extensions\Installed\Importer\Services\DatabaseAdapters;

use PDO;

class OsTicketAdapter extends AbstractDatabaseImportAdapter
{
    protected ?array $operatorIds = null;
    protected array $emailUserCache = [];
    protected ?array $statusMap = null;
    protected ?array $priorityMap = null;
    protected array $tables = [];
    protected bool $kbExtensionEnabled = false;

    public static function id(): string
    {
        return 'osticket';
    }

    public static function label(): string
    {
        return 'osTicket';
    }

    public static function defaults(): array
    {
        return [
            'chunk_size' => 100,
            'import_departments' => true,
            'import_categories' => true,
            'import_users' => true,
            'import_operators' => true,
            'import_tickets' => true,
            'import_responses' => true,
            'import_articles' => true,
        ];
    }

    public static function isPreview(): bool
    {
        return false;
    }

    public function boot(array $config, array $options = []): void
    {
        parent::boot($config, $options);

        $this->initialiseTableMap();
        $this->loadStatusMap();
        $this->loadPriorityMap();
        $this->checkKbExtension();
    }

    public function stages(): array
    {
        return [
            'departments',
            'categories',
            'users',
            'operators',
            'tickets',
            'responses',
            'articles',
        ];
    }

    public function count(string $stage): int
    {
        return match ($stage) {
            'departments' => $this->countDepartments(),
            'categories' => $this->countCategories(),
            'users' => $this->countUsers(),
            'operators' => $this->countOperators(),
            'tickets' => $this->countTickets(),
            'responses' => $this->countResponses(),
            'articles' => $this->countArticles(),
            default => 0,
        };
    }

    public function fetch(string $stage, int $offset, int $limit): array
    {
        return match ($stage) {
            'departments' => $this->fetchDepartments($offset, $limit),
            'categories' => $this->fetchCategories($offset, $limit),
            'users' => $this->fetchUsers($offset, $limit),
            'operators' => $this->fetchOperators($offset, $limit),
            'tickets' => $this->fetchTickets($offset, $limit),
            'responses' => $this->fetchResponses($offset, $limit),
            'articles' => $this->fetchArticles($offset, $limit),
            default => [],
        };
    }

    public function mapStatus(?int $statusId): string
    {
        if ($statusId === null || $this->statusMap === null) {
            return 'open';
        }

        return $this->statusMap[$statusId] ?? 'open';
    }

    public function mapPriority(?int $priorityId): string
    {
        if ($priorityId === null || $this->priorityMap === null) {
            return 'medium';
        }

        return $this->priorityMap[$priorityId] ?? 'medium';
    }

    public function isOperatorUser(int $userId): bool
    {
        if ($this->operatorIds === null) {
            $this->loadOperatorIds();
        }

        return in_array($userId, $this->operatorIds, true);
    }

    public function findUser(int $userId): ?array
    {
        $userTable = $this->resolvedTable('user');
        $userEmailTable = $this->resolvedTable('user_email');

        if ($userTable && $this->tableExists($userTable)) {
            $stmt = $this->connection()->prepare("
                SELECT u.*, ue.address AS email
                FROM {$this->table($userTable)} u
                LEFT JOIN {$this->table($userEmailTable)} ue ON ue.id = u.default_email_id
                WHERE u.id = :id
                LIMIT 1
            ");
            $stmt->execute(['id' => $userId]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($user) {
                return $user;
            }
        }

        $staffTable = $this->resolvedTable('staff');
        if ($staffTable && $this->tableExists($staffTable)) {
            $stmt = $this->connection()->prepare("
                SELECT *
                FROM {$this->table($staffTable)}
                WHERE staff_id = :id
                LIMIT 1
            ");
            $stmt->execute(['id' => $userId]);
            return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
        }

        return null;
    }

    protected function initialiseTableMap(): void
    {
        $prefix = $this->prefix();
        $tableNames = [
            'department' => $prefix . 'department',
            'staff' => $prefix . 'staff',
            'user' => $prefix . 'user',
            'user_email' => $prefix . 'user_email',
            'ticket' => $prefix . 'ticket',
            'ticket_status' => $prefix . 'ticket_status',
            'ticket_priority' => $prefix . 'ticket_priority',
            'thread' => $prefix . 'thread',
            'thread_entry' => $prefix . 'thread_entry',
            'kb_category' => $prefix . 'kb_category',
            'kb_entry' => $prefix . 'kb_entry',
        ];

        foreach ($tableNames as $key => $tableName) {
            if ($this->tableExists($tableName)) {
                $this->tables[$key] = $tableName;
            }
        }
    }

    protected function resolvedTable(string $key): ?string
    {
        return $this->tables[$key] ?? null;
    }

    protected function checkKbExtension(): void
    {
        try {
            $extensionManager = app(\App\Extensions\ExtensionManager::class);
            $this->kbExtensionEnabled = $extensionManager->isInstalled('knowledgebase')
                && $extensionManager->isEnabled('knowledgebase');
        } catch (\Exception $e) {
            $this->kbExtensionEnabled = false;
        }
    }

    protected function loadStatusMap(): void
    {
        $table = $this->resolvedTable('ticket_status');
        if (!$table || !$this->tableExists($table)) {
            $this->statusMap = [];
            return;
        }

        $stmt = $this->connection()->query("
            SELECT id, state FROM {$this->table($table)}
        ");

        $this->statusMap = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
            $state = strtolower(trim((string) ($row['state'] ?? '')));
            $this->statusMap[(int) $row['id']] = match ($state) {
                'closed' => 'closed',
                'open' => 'open',
                'archived' => 'closed',
                'deleted' => 'closed',
                default => 'open',
            };
        }
    }

    protected function loadPriorityMap(): void
    {
        $table = $this->resolvedTable('ticket_priority');
        if (!$table || !$this->tableExists($table)) {
            $this->priorityMap = [];
            return;
        }

        $stmt = $this->connection()->query("
            SELECT priority_id, priority_urgency FROM {$this->table($table)}
        ");

        $this->priorityMap = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
            $urgency = (int) ($row['priority_urgency'] ?? 2);
            $this->priorityMap[(int) $row['priority_id']] = match (true) {
                $urgency === 0 => 'none',
                $urgency === 1 => 'low',
                $urgency === 2 => 'medium',
                $urgency === 3 => 'high',
                $urgency >= 4 => 'emergency',
                default => 'medium',
            };
        }
    }

    protected function loadOperatorIds(): void
    {
        $table = $this->resolvedTable('staff');
        if (!$table || !$this->tableExists($table)) {
            $this->operatorIds = [];
            return;
        }

        $stmt = $this->connection()->query("
            SELECT staff_id FROM {$this->table($table)}
            WHERE isactive = 1
        ");

        $this->operatorIds = array_map(
            fn($row) => (int) $row['staff_id'],
            $stmt->fetchAll(PDO::FETCH_ASSOC)
        );
    }

    protected function resolveUserEmail(int $userId): ?string
    {
        if (isset($this->emailUserCache[$userId])) {
            return $this->emailUserCache[$userId];
        }

        $userTable = $this->resolvedTable('user');
        $userEmailTable = $this->resolvedTable('user_email');

        if (!$userTable || !$userEmailTable) {
            $this->emailUserCache[$userId] = null;
            return null;
        }

        $stmt = $this->connection()->prepare("
            SELECT u.default_email_id, ue.address
            FROM {$this->table($userTable)} u
            LEFT JOIN {$this->table($userEmailTable)} ue ON ue.id = u.default_email_id
            WHERE u.id = :user_id
            LIMIT 1
        ");
        $stmt->execute(['user_id' => $userId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($result && !empty($result['address'])) {
            $email = strtolower(trim($result['address']));
            $this->emailUserCache[$userId] = $email;
            return $email;
        }

        $stmt = $this->connection()->prepare("
            SELECT address FROM {$this->table($userEmailTable)}
            WHERE user_id = :user_id
            ORDER BY id ASC
            LIMIT 1
        ");
        $stmt->execute(['user_id' => $userId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        $email = $result ? strtolower(trim($result['address'])) : null;
        $this->emailUserCache[$userId] = $email;
        return $email;
    }

    protected function countDepartments(): int
    {
        $table = $this->resolvedTable('department');
        if (!$table || !$this->tableExists($table)) {
            return 0;
        }

        $stmt = $this->connection()->query("SELECT COUNT(*) FROM {$this->table($table)}");
        return (int) $stmt->fetchColumn();
    }

    protected function fetchDepartments(int $offset, int $limit): array
    {
        $table = $this->resolvedTable('department');
        if (!$table || !$this->tableExists($table)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT
                id,
                name,
                ispublic,
                signature,
                created,
                updated
            FROM {$this->table($table)}
            ORDER BY id ASC
            LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();

        $results = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
            $results[] = [
                'id' => (int) $row['id'],
                'name' => $row['name'],
                'email' => null,
                'public' => (bool) ($row['ispublic'] ?? true),
                'description' => $row['signature'] ?? null,
                'priority_enabled' => 1,
                'created_at' => $row['created'] ?? null,
                'updated_at' => $row['updated'] ?? null,
            ];
        }

        return $results;
    }

    protected function countCategories(): int
    {
        if (!$this->kbExtensionEnabled) {
            return 0;
        }

        $table = $this->resolvedTable('kb_category');
        if (!$table || !$this->tableExists($table)) {
            return 0;
        }

        $stmt = $this->connection()->query("SELECT COUNT(*) FROM {$this->table($table)}");
        return (int) $stmt->fetchColumn();
    }

    protected function fetchCategories(int $offset, int $limit): array
    {
        if (!$this->kbExtensionEnabled) {
            return [];
        }

        $table = $this->resolvedTable('kb_category');
        if (!$table || !$this->tableExists($table)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT
                category_id AS id,
                name,
                ispublic,
                created,
                updated
            FROM {$this->table($table)}
            ORDER BY category_id ASC
            LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();

        $results = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
            $results[] = [
                'id' => (int) $row['id'],
                'name' => $row['name'],
                'status' => (bool) ($row['ispublic'] ?? true) ? 1 : 0,
                'created_at' => $row['created'] ?? null,
                'updated_at' => $row['updated'] ?? null,
            ];
        }

        return $results;
    }

    protected function countUsers(): int
    {
        $table = $this->resolvedTable('user');
        if (!$table || !$this->tableExists($table)) {
            return 0;
        }

        $stmt = $this->connection()->query("SELECT COUNT(*) FROM {$this->table($table)}");
        return (int) $stmt->fetchColumn();
    }

    protected function fetchUsers(int $offset, int $limit): array
    {
        $userTable = $this->resolvedTable('user');
        $userEmailTable = $this->resolvedTable('user_email');

        if (!$userTable || !$this->tableExists($userTable)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT
                u.id,
                u.name,
                u.default_email_id,
                ue.address AS email,
                u.created,
                u.updated
            FROM {$this->table($userTable)} u
            LEFT JOIN {$this->table($userEmailTable)} ue ON ue.id = u.default_email_id
            ORDER BY u.id ASC
            LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();

        $results = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
            $email = $row['email'] ?: $this->resolveUserEmail((int) $row['id']);
            $email = $email ? strtolower(trim($email)) : null;

            $name = $row['name'] ?: null;
            if (!$name && $email) {
                $name = explode('@', $email)[0];
            }

            $results[] = [
                'id' => (int) $row['id'],
                'name' => $name,
                'email' => $email,
                'confirmed' => 1,
                'company' => null,
                'created_at' => $row['created'] ?? null,
                'updated_at' => $row['updated'] ?? null,
            ];
        }

        return $results;
    }

    protected function countOperators(): int
    {
        $table = $this->resolvedTable('staff');
        if (!$table || !$this->tableExists($table)) {
            return 0;
        }

        $stmt = $this->connection()->query("
            SELECT COUNT(*) FROM {$this->table($table)}
            WHERE isactive = 1
        ");
        return (int) $stmt->fetchColumn();
    }

    protected function fetchOperators(int $offset, int $limit): array
    {
        $table = $this->resolvedTable('staff');
        if (!$table || !$this->tableExists($table)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT
                staff_id,
                username,
                firstname,
                lastname,
                email,
                created,
                updated
            FROM {$this->table($table)}
            WHERE isactive = 1
            ORDER BY staff_id ASC
            LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();

        $results = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
            $results[] = [
                'id' => (int) $row['staff_id'],
                'user_id' => (int) $row['staff_id'],
                'user_email' => strtolower(trim($row['email'])),
                'firstname' => $row['firstname'],
                'lastname' => $row['lastname'],
                'username' => $row['username'],
                'created_at' => $row['created'] ?? null,
                'updated_at' => $row['updated'] ?? null,
            ];
        }

        return $results;
    }

    protected function countTickets(): int
    {
        $table = $this->resolvedTable('ticket');
        if (!$table || !$this->tableExists($table)) {
            return 0;
        }

        $stmt = $this->connection()->query("SELECT COUNT(*) FROM {$this->table($table)}");
        return (int) $stmt->fetchColumn();
    }

    protected function fetchTickets(int $offset, int $limit): array
    {
        $ticketTable = $this->resolvedTable('ticket');
        $threadTable = $this->resolvedTable('thread');
        $threadEntryTable = $this->resolvedTable('thread_entry');

        if (!$ticketTable || !$this->tableExists($ticketTable)) {
            return [];
        }

        $query = "
            SELECT
                t.ticket_id AS id,
                t.number,
                t.user_id,
                t.dept_id AS department_id,
                t.status_id,
                t.priority_id,
                t.staff_id AS assigned_to,
                t.source,
                t.ip_address,
                t.created,
                t.updated,
                t.closed,
                t.lastupdate AS last_reply_at,
                te.title AS subject,
                te.body AS message
            FROM {$this->table($ticketTable)} t
        ";

        if ($threadTable && $threadEntryTable && $this->tableExists($threadTable) && $this->tableExists($threadEntryTable)) {
            $query .= "
                LEFT JOIN {$this->table($threadTable)} th
                    ON th.object_id = t.ticket_id
                    AND th.object_type = 'T'
                LEFT JOIN {$this->table($threadEntryTable)} te
                    ON te.thread_id = th.id
                    AND te.id = (
                        SELECT MIN(te2.id)
                        FROM {$this->table($threadEntryTable)} te2
                        WHERE te2.thread_id = th.id
                    )
            ";
        }

        $query .= "
            ORDER BY t.ticket_id ASC
            LIMIT :limit OFFSET :offset
        ";

        $stmt = $this->connection()->prepare($query);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();

        $results = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
            $results[] = [
                'id' => (int) $row['id'],
                'number' => $row['number'] ?? null,
                'user_id' => (int) $row['user_id'],
                'department_id' => (int) ($row['department_id'] ?? 0),
                'status' => $this->mapStatus((int) ($row['status_id'] ?? 0)),
                'priority' => $this->mapPriority((int) ($row['priority_id'] ?? 0)),
                'subject' => $row['subject'] ?? 'No subject',
                'message' => $row['message'] ?? '',
                'assigned_to' => $row['assigned_to'] ? (int) $row['assigned_to'] : null,
                'source' => $row['source'] ?? null,
                'ip_address' => $row['ip_address'] ?? null,
                'cc_list' => [],
                'assignment_history' => [],
                'created_at' => $row['created'] ?? null,
                'updated_at' => $row['updated'] ?? null,
                'closed_at' => $row['closed'] ?? null,
                'last_reply_at' => $row['last_reply_at'] ?? null,
            ];
        }

        return $results;
    }

    protected function countResponses(): int
    {
        $threadTable = $this->resolvedTable('thread');
        $threadEntryTable = $this->resolvedTable('thread_entry');

        if (!$threadTable || !$threadEntryTable || !$this->tableExists($threadTable) || !$this->tableExists($threadEntryTable)) {
            return 0;
        }

        $stmt = $this->connection()->query("
            SELECT COUNT(*)
            FROM {$this->table($threadEntryTable)} te
            INNER JOIN {$this->table($threadTable)} th ON th.id = te.thread_id
            WHERE th.object_type = 'T'
                AND te.id != (
                    SELECT MIN(te2.id)
                    FROM {$this->table($threadEntryTable)} te2
                    WHERE te2.thread_id = th.id
                )
        ");
        return (int) $stmt->fetchColumn();
    }

    protected function fetchResponses(int $offset, int $limit): array
    {
        $threadTable = $this->resolvedTable('thread');
        $threadEntryTable = $this->resolvedTable('thread_entry');

        if (!$threadTable || !$threadEntryTable || !$this->tableExists($threadTable) || !$this->tableExists($threadEntryTable)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT
                te.id,
                th.object_id AS ticket_id,
                te.staff_id,
                te.user_id,
                te.body,
                te.type,
                te.ip_address,
                te.created,
                te.updated
            FROM {$this->table($threadEntryTable)} te
            INNER JOIN {$this->table($threadTable)} th ON th.id = te.thread_id
            WHERE th.object_type = 'T'
                AND te.id != (
                    SELECT MIN(te2.id)
                    FROM {$this->table($threadEntryTable)} te2
                    WHERE te2.thread_id = th.id
                )
            ORDER BY te.created ASC
            LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();

        $results = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
            $userId = $row['staff_id'] ? (int) $row['staff_id'] : (int) $row['user_id'];
            $type = strtoupper($row['type'] ?? '') === 'N' ? 1 : 0;

            $results[] = [
                'id' => (int) $row['id'],
                'ticket_id' => (int) $row['ticket_id'],
                'user_id' => $userId,
                'purified_text' => $row['body'] ?? '',
                'text' => $row['body'] ?? '',
                'type' => $type,
                'ip_address' => $row['ip_address'] ?? null,
                'created_at' => $row['created'] ?? null,
                'updated_at' => $row['updated'] ?? null,
            ];
        }

        return $results;
    }

    protected function countArticles(): int
    {
        if (!$this->kbExtensionEnabled) {
            return 0;
        }

        $table = $this->resolvedTable('kb_entry');
        if (!$table || !$this->tableExists($table)) {
            return 0;
        }

        $stmt = $this->connection()->query("SELECT COUNT(*) FROM {$this->table($table)}");
        return (int) $stmt->fetchColumn();
    }

    protected function fetchArticles(int $offset, int $limit): array
    {
        if (!$this->kbExtensionEnabled) {
            return [];
        }

        $table = $this->resolvedTable('kb_entry');
        if (!$table || !$this->tableExists($table)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT
                entry_id AS id,
                category_id,
                title,
                body AS content,
                ispublished,
                views,
                created,
                updated
            FROM {$this->table($table)}
            ORDER BY entry_id ASC
            LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();

        $results = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
            $results[] = [
                'id' => (int) $row['id'],
                'category_id' => (int) $row['category_id'],
                'title' => $row['title'],
                'slug' => $row['title'],
                'content' => $row['content'] ?? '',
                'status' => (bool) ($row['ispublished'] ?? false) ? 1 : 0,
                'visibility' => 0,
                'views' => (int) ($row['views'] ?? 0),
                'likes' => 0,
                'tags' => '',
                'created_at' => $row['created'] ?? null,
                'updated_at' => $row['updated'] ?? null,
            ];
        }

        return $results;
    }
}
