<?php

namespace App\Extensions\Installed\Importer\Services\DatabaseAdapters;

class BlestaAdapter extends AbstractDatabaseImportAdapter implements DatabaseImportAdapter
{
    protected array $tables = [];

    protected ?array $operatorIds = null;
    protected array $emailUserCache = [];
    protected array $clientUserMap = [];

    public static function id(): string
    {
        return 'blesta';
    }

    public static function label(): string
    {
        return 'Blesta';
    }

    public static function defaults(): array
    {
        return [
            'chunk_size' => 100,
            'options' => [
                'import_departments' => true,
                'import_categories' => true,
                'import_users' => true,
                'import_operators' => true,
                'import_tickets' => true,
                'import_responses' => true,
                'import_articles' => true,
            ],
        ];
    }

    public static function isPreview(): bool
    {
        return false;
    }

    public function boot(array $dbConfig, array $options = []): void
    {
        parent::boot($dbConfig, $options);
        $this->initialiseTableMap();
    }

    public function stages(): array
    {
        return [
            'departments',
            'categories',
            'users',
            'operators',
            'tickets',
            'responses',
            'articles',
        ];
    }

    public function count(string $stage): int
    {
        return match ($stage) {
            'departments' => $this->countSimple('departments'),
            'categories'  => $this->countSimple('categories'),
            'users'       => $this->countUsers(),
            'operators'   => $this->countSimple('operators'),
            'tickets'     => $this->countSimple('tickets'),
            'responses'   => $this->countSimple('responses'),
            'articles'    => $this->countSimple('articles'),
            default       => 0,
        };
    }

    public function fetch(string $stage, int $offset, int $limit): array
    {
        return match ($stage) {
            'departments' => $this->fetchDepartments($offset, $limit),
            'categories'  => $this->fetchCategories($offset, $limit),
            'users'       => $this->fetchUsers($offset, $limit),
            'operators'   => $this->fetchOperators($offset, $limit),
            'tickets'     => $this->fetchTickets($offset, $limit),
            'responses'   => $this->fetchResponses($offset, $limit),
            'articles'    => $this->fetchArticles($offset, $limit),
            default       => [],
        };
    }

    public function mapStatus(?int $statusId): string
    {
        return match ((int) ($statusId ?? 0)) {
            5 => 'closed',
            4 => 'in progress',
            3 => 'awaiting reply',
            2 => 'open',
            default => 'open',
        };
    }

    public function mapPriority(?int $priorityId): string
    {
        return match ((int) ($priorityId ?? 0)) {
            4 => 'emergency',
            3 => 'high',
            2 => 'medium',
            1 => 'low',
            default => 'none',
        };
    }

    public function isOperatorUser(int $userId): bool
    {
        if ($this->operatorIds === null) {
            $table = $this->resolvedTable('operators');
            if (!$table || !$this->tableExists($table)) {
                $this->operatorIds = [];
            } else {
                $stmt = $this->connection()->query("
                    SELECT id FROM {$this->table($table)}
                ");
                $this->operatorIds = $stmt ? array_map('intval', array_column($stmt->fetchAll(), 'id')) : [];
            }
        }

        return in_array($userId, $this->operatorIds ?? [], true);
    }

    public function findUser(int $userId): ?array
    {
        $clients = $this->resolvedTable('clients');
        $users = $this->resolvedTable('blesta_users');
        if ($clients && $users && $this->tableExists($clients) && $this->tableExists($users)) {
            $stmt = $this->connection()->prepare("
                SELECT c.id AS client_id, u.id AS user_id, u.email, u.first_name, u.last_name, u.date_added AS created_at, u.date_updated AS updated_at
                  FROM {$this->table($clients)} c
                  JOIN {$this->table($users)} u ON u.id = c.user_id
                 WHERE c.id = :id
                 LIMIT 1
            ");
            $stmt->bindValue(':id', $userId, \PDO::PARAM_INT);
            $stmt->execute();
            $row = $stmt->fetch();
            if ($row) {
                return [
                    'id' => (int) $row['client_id'],
                    'email' => $row['email'],
                    'firstname' => $row['first_name'] ?? null,
                    'lastname' => $row['last_name'] ?? null,
                    'created_at' => $row['created_at'] ?? null,
                    'updated_at' => $row['updated_at'] ?? null,
                ];
            }
        }

        if ($users && $this->tableExists($users)) {
            $stmt = $this->connection()->prepare("
                SELECT u.* FROM {$this->table($users)} u WHERE u.id = :id LIMIT 1
            ");
            $stmt->bindValue(':id', $userId, \PDO::PARAM_INT);
            $stmt->execute();
            $row = $stmt->fetch();
            if ($row) {
                return [
                    'id' => (int) $row['id'],
                    'email' => $row['email'] ?? null,
                    'firstname' => $row['first_name'] ?? null,
                    'lastname' => $row['last_name'] ?? null,
                    'created_at' => $row['date_added'] ?? null,
                    'updated_at' => $row['date_updated'] ?? null,
                ];
            }
        }

        return null;
    }

    protected function initialiseTableMap(): void
    {
        $this->tables = [
            'blesta_users' => $this->detectTable(['users', 'blesta_users']),
            'departments' => $this->detectTable(['support_departments']),
            'categories'  => $this->detectTable(['support_kb_categories', 'support_knowledgebase_categories']),
            'articles'    => $this->detectTable(['support_kb_articles', 'support_knowledgebase_articles']),
            'tickets'     => $this->detectTable(['support_tickets']),
            'responses'   => $this->detectTable(['support_tickets_replies', 'support_replies']),
            'operators'   => $this->detectTable(['staff', 'support_staff']),
            'clients'     => $this->detectTable(['clients']),
        ];
    }

    protected function detectTable(array $candidates): ?string
    {
        foreach ($candidates as $candidate) {
            if ($this->tableExists($candidate)) {
                return $candidate;
            }
        }
        return null;
    }

    protected function resolvedTable(string $key): ?string
    {
        return $this->tables[$key] ?? null;
    }

    protected function countSimple(string $key): int
    {
        $table = $this->resolvedTable($key);
        if (!$table || !$this->tableExists($table)) {
            return 0;
        }
        $stmt = $this->connection()->query("SELECT COUNT(*) FROM {$this->table($table)}");
        return (int) $stmt->fetchColumn();
    }

    protected function countUsers(): int
    {
        $clients = $this->resolvedTable('clients');
        if ($clients && $this->tableExists($clients)) {
            return $this->countSimple('clients');
        }
        $users = $this->resolvedTable('blesta_users');
        if ($users && $this->tableExists($users)) {
            return $this->countSimple('blesta_users');
        }
        return 0;
    }

    protected function fetchDepartments(int $offset, int $limit): array
    {
        $table = $this->resolvedTable('departments');
        if (!$table || !$this->tableExists($table)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT * FROM {$this->table($table)} d
             ORDER BY d.id ASC
             LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll() ?: [];

        return array_map(function (array $row): array {
            return [
                'id' => $row['id'] ?? null,
                'name' => $row['name'] ?? ($row['department_name'] ?? null),
                'public' => (int) ($row['public'] ?? ($row['is_public'] ?? 1)) ? 1 : 0,
                'email' => $row['email'] ?? ($row['department_email'] ?? null),
                'priority_enabled' => 1,
                'description' => $row['description'] ?? null,
            ];
        }, $rows);
    }

    protected function fetchCategories(int $offset, int $limit): array
    {
        $table = $this->resolvedTable('categories');
        if (!$table || !$this->tableExists($table)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT * FROM {$this->table($table)} c
             ORDER BY c.id ASC
             LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll() ?: [];

        return array_map(function (array $row): array {
            return [
                'id' => $row['id'] ?? null,
                'name' => $row['name'] ?? ($row['title'] ?? null),
                'status' => isset($row['status']) ? (int) $row['status'] : 1,
            ];
        }, $rows);
    }

    protected function fetchUsers(int $offset, int $limit): array
    {
        $clients = $this->resolvedTable('clients');
        $users = $this->resolvedTable('blesta_users');

        if ($clients && $users && $this->tableExists($clients) && $this->tableExists($users)) {
            $stmt = $this->connection()->prepare("
                SELECT c.id AS id, u.email, u.first_name, u.last_name, u.date_added AS created_at, u.date_updated AS updated_at
                  FROM {$this->table($clients)} c
                  JOIN {$this->table($users)} u ON u.id = c.user_id
                 ORDER BY c.id ASC
                 LIMIT :limit OFFSET :offset
            ");
            $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
            $stmt->execute();
            $rows = $stmt->fetchAll() ?: [];

            foreach ($rows as $r) {
                $this->clientUserMap[(int) $r['id']] = $r['email'] ?? null;
            }

            return array_map(function (array $row): array {
                $email = $this->normalizeEmail($row['email'] ?? null);
                return [
                    'id' => (int) $row['id'],
                    'email' => $email,
                    'name' => $this->normalizeName(trim(($row['first_name'] ?? '') . ' ' . ($row['last_name'] ?? '')), $email),
                    'company' => null,
                    'confirmed' => 1,
                    'created_at' => $row['created_at'] ?? null,
                    'updated_at' => $row['updated_at'] ?? null,
                ];
            }, $rows);
        }

        if ($users && $this->tableExists($users)) {
            $stmt = $this->connection()->prepare("
                SELECT * FROM {$this->table($users)} u
                 ORDER BY u.id ASC
                 LIMIT :limit OFFSET :offset
            ");
            $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
            $stmt->execute();
            $rows = $stmt->fetchAll() ?: [];

            return array_map(function (array $row): array {
                $email = $this->normalizeEmail($row['email'] ?? null);
                return [
                    'id' => (int) $row['id'],
                    'email' => $email,
                    'name' => $this->normalizeName(trim(($row['first_name'] ?? '') . ' ' . ($row['last_name'] ?? '')), $email),
                    'company' => null,
                    'confirmed' => 1,
                    'created_at' => $row['date_added'] ?? null,
                    'updated_at' => $row['date_updated'] ?? null,
                ];
            }, $rows);
        }

        return [];
    }

    protected function fetchOperators(int $offset, int $limit): array
    {
        $operators = $this->resolvedTable('operators');
        $users = $this->resolvedTable('blesta_users');
        if (!$operators || !$this->tableExists($operators)) {
            return [];
        }

        $joinUsers = $this->columnExists($operators, 'user_id') && $users && $this->tableExists($users);

        if ($joinUsers) {
            $stmt = $this->connection()->prepare("
                SELECT s.*, s.id AS id, u.email AS user_email, u.first_name, u.last_name
                  FROM {$this->table($operators)} s
                  LEFT JOIN {$this->table($users)} u ON u.id = s.user_id
                 ORDER BY s.id ASC
                 LIMIT :limit OFFSET :offset
            ");
        } else {
            $stmt = $this->connection()->prepare("
                SELECT s.*, s.id AS id
                  FROM {$this->table($operators)} s
                 ORDER BY s.id ASC
                 LIMIT :limit OFFSET :offset
            ");
        }

        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll() ?: [];

        return array_map(function (array $row): array {
            return [
                'id' => (int) ($row['id'] ?? $row['staff_id'] ?? 0),
                'user_id' => (int) ($row['user_id'] ?? ($row['id'] ?? 0)),
                'user_email' => $row['user_email'] ?? ($row['email'] ?? null),
                'firstname' => $row['first_name'] ?? ($row['firstname'] ?? null),
                'lastname' => $row['last_name'] ?? ($row['lastname'] ?? null),
            ];
        }, $rows);
    }

    protected function fetchTickets(int $offset, int $limit): array
    {
        $tickets = $this->resolvedTable('tickets');
        if (!$tickets || !$this->tableExists($tickets)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT t.* FROM {$this->table($tickets)} t
             ORDER BY t.id ASC
             LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll() ?: [];

        return array_map(function (array $row): array {
            $userId = $row['client_id'] ?? ($row['user_id'] ?? null);
            if ($userId !== null) {
                $userId = (int) $userId;
            } elseif (!empty($row['email'])) {
                $userId = $this->resolveClientIdByEmail((string) $row['email']);
            }

            $status = strtolower((string) ($row['status'] ?? ($row['state'] ?? '')));
            $statusId = $this->statusCodeFromValue($status);

            $priority = strtolower((string) ($row['priority'] ?? ($row['urgency'] ?? '')));
            $priorityId = $this->priorityCodeFromValue($priority);

            $deptId = $row['department_id'] ?? ($row['dept_id'] ?? null);
            $number = $row['code'] ?? ($row['number'] ?? (string) ($row['id'] ?? ''));

            return [
                'id' => (int) ($row['id'] ?? 0),
                'number' => (string) $number,
                'user_id' => $userId,
                'department_id' => $deptId !== null ? (int) $deptId : null,
                'subject' => $row['subject'] ?? ($row['summary'] ?? "Ticket {$row['id']}"),
                'message' => $row['details'] ?? ($row['message'] ?? ($row['content'] ?? '')),
                'status_id' => $statusId,
                'priority_id' => $priorityId,
                'cc_list' => [],
                'assigned_to' => isset($row['staff_id']) ? (int) $row['staff_id'] : null,
                'created_at' => $row['date_added'] ?? ($row['created'] ?? null),
                'updated_at' => $row['date_updated'] ?? ($row['updated'] ?? null),
                'last_reply_at' => $row['last_reply_date'] ?? null,
                'first_staff_reply_at' => null,
                'resolved_at' => $row['date_closed'] ?? null,
                'closed_at' => $row['date_closed'] ?? null,
                'assignment_history' => [],
            ];
        }, $rows);
    }

    protected function fetchResponses(int $offset, int $limit): array
    {
        $responses = $this->resolvedTable('responses');
        $tickets = $this->resolvedTable('tickets');
        if (!$responses || !$tickets || !$this->tableExists($responses) || !$this->tableExists($tickets)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT r.* FROM {$this->table($responses)} r
             ORDER BY r.id ASC
             LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll() ?: [];

        return array_map(function (array $row): array {
            $userId = $row['user_id'] ?? null;
            if ($userId === null && !empty($row['client_id'])) {
                $userId = (int) $row['client_id'];
            }
            if ($userId === null && !empty($row['staff_id'])) {
                $userId = (int) $row['staff_id'];
            }

            $type = (isset($row['type']) ? (int) $row['type'] : 0); // 1 = note
            $isNote = $type === 1;

            return [
                'id' => (int) ($row['id'] ?? 0),
                'ticket_id' => (int) ($row['ticket_id'] ?? ($row['support_ticket_id'] ?? 0)),
                'user_id' => $userId !== null ? (int) $userId : null,
                'purified_text' => $row['message'] ?? ($row['details'] ?? ($row['content'] ?? '')),
                'text' => $row['message'] ?? ($row['details'] ?? ($row['content'] ?? '')),
                'ip_address' => $row['ip_address'] ?? ($row['ip'] ?? null),
                'type' => $isNote ? 1 : 0,
                'created_at' => $row['date_added'] ?? ($row['created'] ?? null),
                'updated_at' => $row['date_updated'] ?? ($row['updated'] ?? null),
            ];
        }, $rows);
    }

    protected function fetchArticles(int $offset, int $limit): array
    {
        $articles = $this->resolvedTable('articles');
        if (!$articles || !$this->tableExists($articles)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT * FROM {$this->table($articles)}
             ORDER BY id ASC
             LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll() ?: [];

        return array_map(function (array $row): array {
            return [
                'id' => (int) ($row['id'] ?? 0),
                'category_id' => (int) ($row['category_id'] ?? ($row['cat_id'] ?? 0)),
                'title' => $row['title'] ?? ($row['name'] ?? null),
                'content' => $row['content'] ?? ($row['article'] ?? ($row['body'] ?? '')),
                'slug' => $row['slug'] ?? ($row['seolink'] ?? ($row['title'] ?? null)),
                'status' => isset($row['published']) ? (int) $row['published'] : (isset($row['status']) ? (int) $row['status'] : 1),
                'visibility' => isset($row['private']) ? (int) $row['private'] : (isset($row['visibility']) ? (int) $row['visibility'] : 0),
                'tags' => $row['tags'] ?? '',
                'views' => (int) ($row['views'] ?? 0),
                'likes' => (int) ($row['likes'] ?? 0),
                'created_at' => $row['date_added'] ?? ($row['created'] ?? null),
                'updated_at' => $row['date_updated'] ?? ($row['updated'] ?? null),
            ];
        }, $rows);
    }

    protected function resolveClientIdByEmail(string $email): ?int
    {
        $normalized = strtolower(trim($email));
        if ($normalized === '') {
            return null;
        }
        if (array_key_exists($normalized, $this->emailUserCache)) {
            return $this->emailUserCache[$normalized];
        }

        $clients = $this->resolvedTable('clients');
        $users = $this->resolvedTable('blesta_users');
        if (!$clients || !$users || !$this->tableExists($clients) || !$this->tableExists($users)) {
            $this->emailUserCache[$normalized] = null;
            return null;
        }

        $stmt = $this->connection()->prepare("
            SELECT c.id
              FROM {$this->table($clients)} c
              JOIN {$this->table($users)} u ON u.id = c.user_id
             WHERE LOWER(u.email) = :email
             LIMIT 1
        ");
        $stmt->bindValue(':email', $normalized);
        $stmt->execute();
        $id = $stmt->fetchColumn();

        $this->emailUserCache[$normalized] = $id !== false ? (int) $id : null;
        return $this->emailUserCache[$normalized];
    }

    protected function statusCodeFromValue(mixed $value): int
    {
        if ($value === null) {
            return 2;
        }
        if (is_int($value)) {
            return $value;
        }

        $normalized = strtolower(trim((string) $value));
        return match ($normalized) {
            'closed', 'resolved' => 5,
            'awaiting reply', 'awaiting', 'client reply', 'client-reply', 'customer reply', 'pending' => 3,
            'in progress', 'on hold', 'hold' => 4,
            'new', 'open' => 2,
            default => 2,
        };
    }

    protected function priorityCodeFromValue(mixed $value): int
    {
        if ($value === null) {
            return 0;
        }
        if (is_int($value)) {
            return $value;
        }

        $normalized = strtolower(trim((string) $value));
        return match ($normalized) {
            'low' => 1,
            'medium', 'normal' => 2,
            'high' => 3,
            'urgent', 'critical' => 4,
            default => 0,
        };
    }
}
