<?php

namespace App\Extensions\Concerns;

use App\Models\Extension;
use Illuminate\Support\Facades\File;

/**
 * Trait for extensions to check if other extensions are installed/enabled.
 *
 * This enables inter-extension communication and integration.
 */
trait InteractsWithExtensions
{
    /**
     * Check if an extension is installed.
     */
    public function isExtensionInstalled(string $extensionId): bool
    {
        return Extension::where('extension_id', $extensionId)
            ->where('installed', true)
            ->exists();
    }

    /**
     * Check if an extension is enabled.
     */
    public function isExtensionEnabled(string $extensionId): bool
    {
        return Extension::where('extension_id', $extensionId)
            ->where('installed', true)
            ->where('enabled', true)
            ->exists();
    }

    /**
     * Get an extension instance if it's installed and enabled.
     */
    public function getExtension(string $extensionId): ?object
    {
        if (!$this->isExtensionEnabled($extensionId)) {
            return null;
        }

        // Load the extension class
        $extensionPath = app_path("Extensions/Installed");
        $directories = File::directories($extensionPath);

        foreach ($directories as $directory) {
            $manifestPath = $directory . '/extension.json';

            if (File::exists($manifestPath)) {
                $manifest = json_decode(File::get($manifestPath), true);

                if ($manifest['id'] === $extensionId) {
                    // Find the extension class
                    $className = str($extensionId)->studly() . 'Extension';
                    $classFile = $directory . '/' . $className . '.php';

                    if (File::exists($classFile)) {
                        $fullClassName = "App\\Extensions\\Installed\\" . basename($directory) . "\\" . $className;

                        if (class_exists($fullClassName)) {
                            return new $fullClassName($directory);
                        }
                    }
                }
            }
        }

        return null;
    }

    /**
     * Call a method on another extension if it's enabled.
     */
    public function callExtensionMethod(string $extensionId, string $method, array $params = [])
    {
        $extension = $this->getExtension($extensionId);

        if ($extension && method_exists($extension, $method)) {
            return call_user_func_array([$extension, $method], $params);
        }

        return null;
    }

    /**
     * Get data from another extension's model.
     */
    public function getExtensionModel(string $extensionId, string $modelName): ?string
    {
        if (!$this->isExtensionEnabled($extensionId)) {
            return null;
        }

        $modelClass = "App\\Extensions\\Installed\\" . str($extensionId)->studly() . "\\Models\\" . $modelName;

        return class_exists($modelClass) ? $modelClass : null;
    }
}
