<?php

namespace App\Console\Commands\Extensions;

use App\Models\Extension;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class EnableCommand extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'extension:enable {id : The extension ID to enable}';

    /**
     * The console command description.
     */
    protected $description = 'Enable an extension';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $extensionId = $this->argument('id');

        $this->info("Enabling extension: {$extensionId}");

        try {
            // Find extension
            $extension = Extension::where('extension_id', $extensionId)->first();

            if (!$extension) {
                $this->error("Extension '{$extensionId}' not found!");
                $this->line('Run: php artisan extension:list to see installed extensions');
                return 1;
            }

            if (!$extension->installed) {
                $this->error("Extension '{$extensionId}' is not installed!");
                $this->line("Run: php artisan extension:install {$extensionId}");
                return 1;
            }

            if ($extension->enabled) {
                $this->warn("Extension '{$extensionId}' is already enabled!");
                return 0;
            }

            // Load extension class
            $extensionClass = $this->findExtensionClass($extensionId);
            if (!$extensionClass) {
                $this->error("Extension class not found for '{$extensionId}'");
                return 1;
            }

            // Get extension path
            $extensionPath = $this->getExtensionPath($extensionId);

            // Instantiate extension with path
            $instance = new $extensionClass($extensionPath);

            // Set extension context
            $reflection = new \ReflectionClass($instance);
            $property = $reflection->getProperty('extension');
            $property->setAccessible(true);
            $property->setValue($instance, $extension);

            // Check requirements
            $this->info('Checking requirements...');
            $requirementCheck = $instance->checkRequirements();

            // Display requirements
            if (!empty($requirementCheck['errors'])) {
                $this->error('❌ Requirements not met:');
                foreach ($requirementCheck['errors'] as $error) {
                    $this->line("  • {$error}");
                }
                $this->newLine();
                $this->error("Cannot enable extension '{$extensionId}' with unmet requirements.");
                $this->newLine();
                $this->info('Please resolve the issues above before enabling the extension.');
                return 1;
            }

            if (!empty($requirementCheck['warnings'])) {
                $this->warn('⚠️  Warnings:');
                foreach ($requirementCheck['warnings'] as $warning) {
                    $this->line("  • {$warning}");
                }
                $this->newLine();
                $this->warn('Continuing with warnings...');
            } else {
                $this->info('✓ All requirements met');
                $this->newLine();
            }

            // Run enable method
            $this->info('Running enable process...');
            $instance->enable();

            // Mark as enabled
            $extension->update(['enabled' => true]);

            $this->newLine();
            $this->info("✓ Extension '{$extensionId}' enabled successfully!");

            return 0;

        } catch (\Exception $e) {
            $this->error('Enable failed: ' . $e->getMessage());
            $this->newLine();
            $this->error('Stack trace:');
            $this->line($e->getTraceAsString());
            return 1;
        }
    }

    /**
     * Find the extension class file.
     */
    protected function findExtensionClass(string $extensionId): ?string
    {
        // Convert kebab-case to StudlyCase (e.g., live-chat -> LiveChat)
        $className = str_replace('-', '', ucwords($extensionId, '-'));

        $path = app_path("Extensions/Installed/{$className}/{$className}Extension.php");

        if (!File::exists($path)) {
            return null;
        }

        $class = "App\\Extensions\\Installed\\{$className}\\{$className}Extension";

        if (!class_exists($class)) {
            return null;
        }

        return $class;
    }

    /**
     * Get the extension directory path.
     */
    protected function getExtensionPath(string $extensionId): string
    {
        // Convert kebab-case to StudlyCase (e.g., live-chat -> LiveChat)
        $className = str_replace('-', '', ucwords($extensionId, '-'));
        return app_path("Extensions/Installed/{$className}");
    }
}
